package clustersim;


import java.util.ArrayList;

import org.apache.commons.math3.distribution.LogNormalDistribution;

import rsalgos.AsynchcronousRSalgorithm;
import rsalgos.RSalgorithm;
import sim.engine.SimState;
import simtools.RandomStream;
import simtools.RandomStreamFactory;

/**
 * Virtual computing cluster emulator
 * @author Bogumil Kaminski & Przemyslaw Szufel
 */

@SuppressWarnings("serial")
public class CompClusterEmulator extends SimState {

	public final int N; //number of points for R&S problem
	public final int numWorkers; //number of workers
	
	RSpoint points[]; //points for the R&S problem	
	Worker workers[]; //workers that evaluate the points
	
	private int optimalSolutionIndex;
	private void updateOptimalSolutionDistanceHistory () {
		double estOptValuel = points[0].getMeanY();
		int estOptIx  = 0;
		double mean;
		for (int i=1;i<N;i++) {
			mean = points[i].getMeanY();
			if (mean > estOptValuel) {
				estOptValuel = mean;
				estOptIx = i;
			}
		}		
		optimalSolutionDistanceBinHistory.add((optimalSolutionIndex==estOptIx)?0:1);
		optimalSolutionDistanceDoubleHistory.add(points[optimalSolutionIndex].u-points[estOptIx].u);
	}
	
	private final RSalgorithm pointSelector; // R&S algrithm
	private RandomStreamFactory randomStreams;
	private RandomStream random;
	
	private ArrayList<Integer> pointSelectionHistory;
	private ArrayList<Integer> optimalSolutionDistanceBinHistory;
	private ArrayList<Double>  optimalSolutionDistanceDoubleHistory;
	boolean debug;
	public CompClusterEmulator(long seed, int N, int numWorkers,RSalgorithm pointSelector, boolean debug) {
		super(seed);	
		this.N = N;
		this.numWorkers = numWorkers;
		this.pointSelector = pointSelector;
		this.debug = debug;
		if (numWorkers > 1 && ! (pointSelector instanceof AsynchcronousRSalgorithm)) {
			throw new Error("if numWorkers>1 job selector needs to be Asynchornous");
		}
		randomStreams = new RandomStreamFactory(N+numWorkers, seed);
		int ids[] = new int[numWorkers];
		for (int i=0;i<numWorkers;i++) ids[i] = N+i;
		random = randomStreams.getStream(ids); 
		pointSelectionHistory = new ArrayList<>();
		optimalSolutionDistanceBinHistory = new ArrayList<>();
		optimalSolutionDistanceDoubleHistory = new ArrayList<>();		
		points = new RSpoint[N];		
		double optSolV = -1e99;
		for (int i=0;i<N;i++) {
			points[i] = new RSpoint(randomStreams.getStream(i),i);
			if (points[i].u > optSolV) {
				optSolV = points[i].u;
				optimalSolutionIndex = i;
			}
		}		
	}	
	@Override public void start() {
		super.start();
		workers = new Worker[numWorkers];
		for (int w=0;w<numWorkers;w++) {
			workers[w] = new Worker(w);
			this.schedule.scheduleOnce(workers[w],w);
		}		
	}
	
	LogNormalDistribution distComputingTime = new LogNormalDistribution(0, 1); //computing time
	
	public void dispatchMainLoop (Worker worker) {
		//this simulates what happens in the main loop
		RSpoint newPoint;
		if (worker.job == null) {
			if (optimalSolutionDistanceBinHistory.size()==0) updateOptimalSolutionDistanceHistory();
			newPoint = pointSelector.getPointInit_k_0(points,worker.w,pointSelectionHistory.size());			
		} else {
			double y = worker.job.get_y(this); //worker output
			worker.job.s--;
			updateOptimalSolutionDistanceHistory();
			newPoint = pointSelector.getNextPoint (y, worker.job, points,worker.w,pointSelectionHistory.size(),debug); 
		}		
		worker.job = newPoint; // assign the selected job
		worker.job.s++;
		pointSelectionHistory.add(newPoint.i);		
		double computingTime = distComputingTime.inverseCumulativeProbability(random.random());
		schedule.scheduleOnce(schedule.getTime()+computingTime, worker.w,worker);
	}
	
	public String getPointSelectionHistory () {
		StringBuilder res = new StringBuilder();
		for (int i=0;i<pointSelectionHistory.size();i++) {
			res.append(pointSelectionHistory.get(i)+((i<pointSelectionHistory.size()-1)?",":""));
		}
		return res.toString();
	}
	
	public String getOptimalSolutionDistanceBinHistory () {
		StringBuilder res = new StringBuilder();
		for (int i=0;i<optimalSolutionDistanceBinHistory.size();i++) {
			res.append(optimalSolutionDistanceBinHistory.get(i)+((i<optimalSolutionDistanceBinHistory.size()-1)?",":""));
		}
		return res.toString();
	}
	
	public String getOptimalSolutionDistanceDoubleHistory () {
		StringBuilder res = new StringBuilder();
		for (int i=0;i<optimalSolutionDistanceDoubleHistory.size();i++) {
			res.append(optimalSolutionDistanceDoubleHistory.get(i)+((i<optimalSolutionDistanceDoubleHistory.size()-1)?",":""));
		}
		return res.toString();
	}
	
	public int getCalculationCount () {
		return pointSelectionHistory.size();
	}
	
	/**
	 * Method for conveniently running this model
	 * @param model
	 * @param schedulerTimeBudget
	 * @param echo
	 */
	public static void run (final CompClusterEmulator model, long pointCalculationsBudget, boolean echo) {
		model.start();
		long lastStepTime = System.currentTimeMillis();
		long startTime = System.currentTimeMillis();
		long reportEvery = 100;
		boolean moreSteps;
		long step = 0;
		double simTime = 0;
		while (moreSteps = model.schedule.step(model)) {
			simTime = model.schedule.getTime();
			step = model.schedule.getSteps();
			
			if (model.getCalculationCount() >= pointCalculationsBudget)
				break;
			if (step % reportEvery == 0) {
				if (echo)
					System.err.println("Step="+step+" schedule.getTime()="+((long)(simTime*10))/10.0+" speed steps/s="+(reportEvery*1000.0/(System.currentTimeMillis()-lastStepTime)));
				lastStepTime = System.currentTimeMillis();
			}
		}
		model.finish();
		model.kill();
		if (echo) {
			System.out.flush();
			if (!moreSteps) 
				System.err.print("The simulation has been exhausted!");
			else 
				System.err.print("Simulation completed!");
			System.err.println(" step="+step+" schedule.getTime()="+((long)(simTime*10))/10.0+" speed steps/s="+(reportEvery*1000.0/(System.currentTimeMillis()-startTime)));
		}
		
	}
	


}

